//
//  ContentView.swift
//  TestDeeplinkApp
//
//  Created by Dimitar Lazarov on 12/01/2025.
//

import SwiftUI
import Combine

final class ViewModel: ObservableObject {
  @Published var transactionAmount: String = ""
  @Published var originalTransactionId: String = ""
  @Published var transactionType: String = "sale"
  
  @Published var deeplinkValue: String = ""
  
  struct Keys {
    static let transactionAmount = "{transaction_amount}"
    static let transactionType = "{transaction_type}"
    static let originalTranId = "{original_tran_id}"
  }
  
  static let decimalSeparator = Locale.current.decimalSeparator!
  static let decimalPlaces: Int = 2
  
  let deeplinkTemplate: String = "softpos://softpos/transaction?payload={\"amount\":\(Keys.transactionAmount),\"responseUrl\":\"testapp://response\",\"type\":\"\(Keys.transactionType)\",\"originalTranId\":\"\(Keys.originalTranId)\",\"currency\":\"EUR\"}"
  
  init() {
    Publishers.CombineLatest3(
      $transactionAmount,
      $transactionType,
      $originalTransactionId
    )
    .throttle(for: 0.5, scheduler: DispatchQueue.main, latest: true)
    .map { amount, type, originalTranId in
      var deeplink: String = self.deeplinkTemplate
      
      deeplink = deeplink.replacingOccurrences(of: ViewModel.Keys.transactionType, with: type)
      
      let minorUnits = Self.minorUnits(from: amount)
      
      deeplink = deeplink.replacingOccurrences(of: ViewModel.Keys.transactionAmount, with: minorUnits)
      
      deeplink = deeplink.replacingOccurrences(of: ViewModel.Keys.originalTranId, with: originalTranId)
      
      return deeplink
    }
    .assign(to: &$deeplinkValue)
  }
  
  static func minorUnits(from amount: String) -> String {
    let items = amount.split(separator: self.decimalSeparator)
    
    var minorUnits: String = items.count > 1 ? String(items.last!.prefix(decimalPlaces)) : ""
    
    if minorUnits.count < decimalPlaces {
      minorUnits += String(repeating: "0", count: decimalPlaces - minorUnits.count)
    }
    
    if let amount = items.first {
      minorUnits = amount + minorUnits
    }
    
    // Remove leading zeros
    minorUnits = Int(minorUnits)!.description
    
    return minorUnits
  }
  
  func updateAmount(_ amount: String) {
    var newValue = amount
    
    newValue.removeAll(where: { !$0.isNumber && String($0) != Self.decimalSeparator })
    
    let items = newValue.split(separator: Self.decimalSeparator)
    
    if items.count > 1 {
      transactionAmount = String(items.first!) + Self.decimalSeparator + String(items.last!.prefix(Self.decimalPlaces))
    }
    else {
      transactionAmount = newValue
    }
  }
  
  func updateOriginalTransactionId(_ value: String) {
    var newValue = value
    newValue.removeAll(where: { !$0.isNumber })
    originalTransactionId = newValue
  }
}

struct ContentView: View {
  @Environment(\.openURL) var openURL
  
  @StateObject private var viewModel: ViewModel = ViewModel()
  @State private var jsonResult: String?
  
  var body: some View {
    VStack {
      Text("SoftPOS Deeplink Test App")
        .font(.largeTitle)
        .multilineTextAlignment(.center)
        .padding()
      Picker("Transaction type", selection: $viewModel.transactionType) {
        Text("Sale").tag("sale")
        Text("Refund").tag("refund")
      }
      .pickerStyle(.segmented)
      .onChange(of: viewModel.transactionType) { newValue in
        if newValue != "refund" {
          viewModel.originalTransactionId = ""
        }
      }
      HStack {
        Text("€")
        TextField("0.00", text: $viewModel.transactionAmount)
          .keyboardType(.decimalPad)
          .onChange(of: viewModel.transactionAmount) { newValue in
            viewModel.updateAmount(newValue)
          }
        Spacer()
      }
      .padding()
      .font(.largeTitle)
      .overlay {
        RoundedRectangle(cornerRadius: 8)
          .inset(by: 0.5)
          .stroke(Color(red: 0.85, green: 0.85, blue: 0.85), lineWidth: 1)
      }
      
      if viewModel.transactionType == "refund" {
        TextField("Original transaction ID", text: $viewModel.originalTransactionId)
          .keyboardType(.decimalPad)
          .onChange(of: viewModel.originalTransactionId) { newValue in
            viewModel.updateOriginalTransactionId(newValue)
          }
          .padding()
          .font(.largeTitle)
          .overlay {
            RoundedRectangle(cornerRadius: 8)
              .inset(by: 0.5)
              .stroke(Color(red: 0.85, green: 0.85, blue: 0.85), lineWidth: 1)
          }
      }
      
      Text(viewModel.deeplinkValue)
        .padding()
      
      Button("Open Deeplink") {
        guard let url = URL(string: viewModel.deeplinkValue) else { return }
        openURL(url)
      }
      .padding()
      .buttonStyle(.borderedProminent)
      
      Spacer()
    }
    .sheet(item: $jsonResult, content: { result in
      TransactionResultView(jsonResult: result)
    })
    .onOpenURL { url in
      guard let components = URLComponents(url: url, resolvingAgainstBaseURL: true),
            let jsonString = components.queryItems?.first(where: {$0.name == "result"})?.value else { return }
      
      do {
        let object = try JSONSerialization.jsonObject(with: jsonString.data(using: .utf8)!, options: []) as? [String: Any]
        let data = try JSONSerialization.data(withJSONObject: object ?? [:], options: .prettyPrinted)
        jsonResult = String(data: data, encoding: .utf8)!
      }
      catch {
        print("Error parsing JSON: \(jsonString) error: \(error)")
      }
      
    }
    .padding()
    .contentShape(Rectangle())
    .gesture(
      DragGesture(minimumDistance: 20)
        .onEnded { _ in
          hideKeyboard()
        }
    )
  }
  
  private func hideKeyboard() {
    UIApplication.shared.sendAction(#selector(UIResponder.resignFirstResponder), to: nil, from: nil, for: nil)
  }
}

#Preview {
  ContentView()
}

extension String: @retroactive Identifiable {
  public var id: Self { self }
}

struct TransactionResultView: View {
  
  @Environment(\.dismiss) private var dismiss
  @State var jsonResult: String = ""
  
  var body: some View {
    VStack {
      Text("Transaction Result")
        .font(.headline)
        .padding()
      TextEditor(text: .constant(jsonResult))
      Spacer()
      Button("Dismiss") {
        dismiss()
      }
    }
  }
}
